### 4. Using Abilities (`wp_get_ability`, `wp_get_abilities`)

Once abilities are registered, they can be retrieved and executed using global functions from the Abilities API.

**Getting a Specific Ability (`wp_get_ability`)**

To get a single ability object by its name (namespace/ability-name):

```php
/**
 * Retrieves a registered ability using Abilities API.
 *
 * @param string $name The name of the registered ability, with its namespace.
 * @return ?WP_Ability The registered ability instance, or null if it is not registered.
 */
function wp_get_ability( string $name ): ?WP_Ability

// Example:
$site_info_ability = wp_get_ability( 'my-plugin/get-site-info' );

if ( $site_info_ability ) {
	// Ability exists and is registered
	$result = $site_info_ability->execute();
	if ( is_wp_error( $site_info ) ) {
		// Handle WP_Error
		echo 'Error: ' . $site_info->get_error_message();
	} else {
		// Use $site_info array
		echo 'Site Name: ' . $site_info['name'];
	}
} else {
	// Ability not found or not registered
}
```

**Getting All Registered Abilities (`wp_get_abilities`)**

To get an array of all registered abilities:

```php
/**
 * Retrieves all registered abilities using Abilities API.
 *
 * @return WP_Ability[] The array of registered abilities.
 */
function wp_get_abilities(): array

// Example: Get all registered abilities
$all_abilities = wp_get_abilities();

foreach ( $all_abilities as $name => $ability ) {
    echo 'Ability Name: ' . esc_html( $ability->get_name() ) . "\n";
    echo 'Label: ' . esc_html( $ability->get_label() ) . "\n";
    echo 'Description: ' . esc_html( $ability->get_description() ) . "\n";
    echo "---\n";
}
```

**Executing an Ability (`$ability->execute()`)**

Once you have a `WP_Ability` object (usually from `wp_get_ability`), you execute it using the `execute()` method.

```php
/**
 * Executes the ability after input validation and running a permission check.
 *
 * @param array $input Optional. The input data for the ability.
 * @return mixed|WP_Error The result of the ability execution, or WP_Error on failure.
 */
// public function execute( array $input = array() )

// Example 1: Ability with no input parameters
$ability = wp_get_ability( 'my-plugin/get-site-info' );
if ( $ability ) {
    $site_info = $ability->execute(); // No input required
    if ( is_wp_error( $site_info ) ) {
        // Handle WP_Error
        echo 'Error: ' . $site_info->get_error_message();
    } else {
        // Use $site_info array
        echo 'Site Name: ' . $site_info['name'];
    }
}

// Example 2: Ability with input parameters
$ability = wp_get_ability( 'my-plugin/update-option' );
if ( $ability ) {
    $input = array(
        'option_name'  => 'blogname',
        'option_value' => 'My Updated Site Name',
    );

    $result = $ability->execute( $input );
    if ( is_wp_error( $result ) ) {
        // Handle WP_Error
        echo 'Error: ' . $result->get_error_message();
    } else {
        // Use $result
        if ( $result['success'] ) {
            echo 'Option updated successfully!';
            echo 'Previous value: ' . $result['previous_value'];
        }
    }
}

// Example 3: Ability with complex input validation
$ability = wp_get_ability( 'my-plugin/send-email' );
if ( $ability ) {
    $input = array(
        'to'      => 'user@example.com',
        'subject' => 'Hello from WordPress',
        'message' => 'This is a test message from the Abilities API.',
    );

    $result = $ability->execute( $input );
    if ( is_wp_error( $result ) ) {
        // Handle WP_Error
        echo 'Error: ' . $result->get_error_message();
    } elseif ( $result['sent'] ) {
        echo 'Email sent successfully!';
    } else {
        echo 'Email failed to send.';
    }
}
```

**Checking Permissions (`$ability->has_permission()`)**

Before executing an ability, you can check if the current user has permission:

```php
$ability = wp_get_ability( 'my-plugin/update-option' );
if ( $ability ) {
    $input = array(
        'option_name'  => 'blogname',
        'option_value' => 'New Site Name',
    );
    
    // Check permission before execution
    if ( $ability->has_permission( $input ) ) {
        $result = $ability->execute( $input );
        if ( is_wp_error( $result ) ) {
	        // Handle WP_Error
	        echo 'Error: ' . $result->get_error_message();
        } else {
        	// Use $result
			if ( $result['success'] ) {
				echo 'Option updated successfully!';
				echo 'Previous value: ' . $result['previous_value'];
			}
        }
    } else {
        echo 'You do not have permission to execute this ability.';
    }
}
```

**Inspecting Ability Properties**

The `WP_Ability` class provides several getter methods to inspect ability properties:

```php
$ability = wp_get_ability( 'my-plugin/get-site-info' );
if ( $ability ) {
    // Basic properties
    echo 'Name: ' . $ability->get_name() . "\n";
    echo 'Label: ' . $ability->get_label() . "\n";
    echo 'Description: ' . $ability->get_description() . "\n";
    
    // Schema information
    $input_schema = $ability->get_input_schema();
    $output_schema = $ability->get_output_schema();
    
    echo 'Input Schema: ' . json_encode( $input_schema, JSON_PRETTY_PRINT ) . "\n";
    echo 'Output Schema: ' . json_encode( $output_schema, JSON_PRETTY_PRINT ) . "\n";
    
    // Metadata
    $meta = $ability->get_meta();
    if ( ! empty( $meta ) ) {
        echo 'Metadata: ' . json_encode( $meta, JSON_PRETTY_PRINT ) . "\n";
    }
}
```

**Error Handling Patterns**

The Abilities API uses several error handling mechanisms:

```php
$ability = wp_get_ability( 'my-plugin/some-ability' );

if ( ! $ability ) {
    // Ability not registered
    echo 'Ability not found';
    return;
}

$result = $ability->execute( $input );

// Check for WP_Error (validation, permission, or callback errors)
if ( is_wp_error( $result ) ) {
    echo 'WP_Error: ' . $result->get_error_message();
    return;
}

// Check for null result (permission denied, invalid callback, or validation failure)
if ( is_null( $result ) ) {
    echo 'Execution returned null - check permissions and callback validity';
    return;
}

// Success - use the result
// Process $result based on the ability's output schema    
```
